using System;
using System.IO;
using System.Text;
using System.Data;
using System.Collections;
namespace gov.va.med.vbecs.DAL.HL7.OpenLibrary.Messages
{

		#region Header

		//<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
		//<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
		//<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
		//<Developers>
		//	<Developer>Brian    lin</Developer>
		//</Developers>
		//<SiteName>Hines OIFO</SiteName>
		//<CreationDate>5/10/2004</CreationDate>
		//<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
		//<summary>The HL7OmgMessage type represents the HL7 v2.4 General Clinical Order Message(OMG). It implements the HL7ProtocolMessage type and is not meant to be inherited.</summary>

		#endregion

	/// <summary>
	/// Class HL7OmgMessage
	/// </summary>
	public sealed class HL7OmgMessage : HL7ProtocolMessage
	{
		/// <summary>
		/// General Clinical Order Message(OMG)
		/// </summary>
		private const string MESSAGE_TYPE = "OMG^O19";

		private const bool RESPONSE_REQUIRED_INDICATOR = true;

		private char[] _delimiters;
		private string[] _msh;
		private string[] _pid;
		private string[] _pv1;
		private string[] _orc;
		private string[] _obr;
		private string[] _dg1;
		private ArrayList _nte;
		private string[] _obx;
		private string _orderControlCode;
		private string[] _universalServiceID;
		private string[] _timingQuantity;

		/// <summary>
		/// Empty static constructor added to get rid of "beforefieldinit" attribute generated by compiler.
		/// </summary>
		static HL7OmgMessage() {}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/15/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7050"> 
		///		<ExpectedInput>Input string</ExpectedInput>
		///		<ExpectedOutput>HL7OmgMessage object</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7051"> 
		///		<ExpectedInput>NULL message input string</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Constructor to load HL7 message.
		/// </summary>
		/// <param name="message">String representation of the HL7 message.</param>
		public HL7OmgMessage( string message )
			: base( message )
		{
			if( message == null )
				throw( new ArgumentNullException( "message" ) );

			_message = message;
			_messageControlID = HL7Utility.GetMessageControlID( message );
			LoadDataFromHL7Message();

		}

		private void LoadDataFromHL7Message()
		{
			_delimiters = HL7Utility.ParseGetMessageDelimiters(this.Message);

			string[] _hl7Segments = HL7Utility.ParseGetAllMessageSegments( this.Message );
			for (int i = 0 ; i < _hl7Segments.Length - 1 ; i++ )
			{
				string[] seg = _hl7Segments[i].Split(_delimiters[0]);
				switch(seg[0])
				{
                    //CR 3243 adding MSH because it is required
                    case "MSH":
                        _msh = _hl7Segments[i].Split(_delimiters[0]);
				        break;

					case "PID":
						_pid = _hl7Segments[i].Split(_delimiters[0]);						
						break;
					case "PV1":
						_pv1 = _hl7Segments[i].Split(_delimiters[0]);
						break;

					case "ORC":
						_orc = _hl7Segments[i].Split(_delimiters[0]);
						break;

					case "OBR":
						_obr = _hl7Segments[i].Split(_delimiters[0]);
						break;

					case "DG1":
						_dg1 = _hl7Segments[i].Split(_delimiters[0]);
						break;

                    case "OBX":
						_obx = _hl7Segments[i].Split(_delimiters[0]);
						break;

					default:
						break;
				}
			}
			_nte = HL7Utility.ParseGetOrderComments( this.Message );
			_orderControlCode = HL7Utility.ConvertString( _orc[1] );
			if( _obr.Length >= 27 )
			{
				_timingQuantity = HL7Utility.ParseGetStringArray( _obr[27], _delimiters[1].ToString() );
			}
			
			_universalServiceID =  HL7Utility.ParseGetStringArray( HL7Utility.ConvertString( _obr[4] ), _delimiters[1].ToString() );

		}

		/// <summary>
		/// WriteHL7MessageBody
		/// </summary>
		/// <param name="writer"></param>
		protected override void WriteHL7MessageBody( StreamWriter writer )
		{
			if( writer == null )
				throw( new ArgumentNullException( "writer" ) );

			writer.Write( this.Message );
			
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7077"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7078"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This method always returns true for this type of message.
		/// </summary>
		/// <returns>boolean</returns>
		public override bool IsResponseRequired()
		{
			return RESPONSE_REQUIRED_INDICATOR;
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7079"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>Message string used as imput message in constructor.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7080"> 
		///		<ExpectedInput>No Failure conditions apply.</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns the full HL7 message.
		/// </summary>
		/// <returns>string representation of HL7 message input.</returns>
		public override string GetMessage()
		{
			return Message;
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7081"> 
		///		<ExpectedInput>Random number in Message Control ID field in HL7 OMG message.</ExpectedInput>
		///		<ExpectedOutput>Same number as Message Control ID in input message.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7082"> 
		///		<ExpectedInput>No failure conditions</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns the Message Control ID associated with this message
		/// </summary>
		/// <returns>Message Control ID in MSH segment</returns>
		public override string GetMessageControlID()
		{
			return MessageControlID;
		}


		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="304"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="306"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Overriden method to get the Message Type.
		/// </summary>
		public override string GetMessageType()
		{
			return MessageType;
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="310"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="314"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Message Type.
		/// </summary>
		public string MessageType
		{
			get
			{
				return MESSAGE_TYPE;
			}
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="315"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="318"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Full HL7 message
		/// </summary>
		public string Message
		{
			get
			{
				return _message;
			}
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="319"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="324"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Message Control ID
		/// </summary>
		public string MessageControlID
		{
			get
			{
				return _messageControlID;
			}
			set
			{
				_messageControlID = value;
			}
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="325"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="327"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Order control code
		/// </summary>
		public string OrderControlCode
		{
			get
			{
				return _orderControlCode;
			}
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="328"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="329"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Message Delimiters
		/// </summary>
		public char[] Delimiters
		{
			get
			{
				return _delimiters;
			}
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="330"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string array</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="331"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// MSH Segment
		/// </summary>
		public string[] MSH
		{
			get
			{
				return _msh;
			}
			set
			{
				_msh = value;
			}
		}


		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="332"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string array</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="333"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// PID Segment
		/// </summary>
		public string[] PID
		{
			get
			{
				return _pid;
			}
			set
			{
				_pid = value;
			}
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="334"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string array</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="335"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// PV1 Segment
		/// </summary>
		public string[] PV1
		{
			get
			{
				return _pv1;
			}
			set
			{
				_pv1 = value;
			}
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="340"> 
		///		<ExpectedInput>string array</ExpectedInput>
		///		<ExpectedOutput>string array</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="341"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// ORC Segment
		/// </summary>
		public string[] ORC
		{
			get
			{
				return _orc;
			}
			set
			{
				_orc = value;
			}
		}


		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="350"> 
		///		<ExpectedInput>string array</ExpectedInput>
		///		<ExpectedOutput>string array</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="351"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// OBR Segment
		/// </summary>
		public string[] OBR
		{
			get
			{
				return _obr;
			}
			set
			{
				_obr = value;
			}
		}


		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="356"> 
		///		<ExpectedInput>string array</ExpectedInput>
		///		<ExpectedOutput>string array</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="357"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// DG1 Segment
		/// </summary>
		public string[] DG1
		{
			get
			{
				return _dg1;
			}
			set
			{
				_dg1 = value;
			}
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="358"> 
		///		<ExpectedInput>string array</ExpectedInput>
		///		<ExpectedOutput>string array</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="361"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// NTE Segment
		/// </summary>
		public ArrayList NTE
		{
			get
			{
				return _nte;
			}
			set
			{
				_nte = value;
			}
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="362"> 
		///		<ExpectedInput>string array</ExpectedInput>
		///		<ExpectedOutput>string array</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="365"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// OBX Segment
		/// </summary>
		public string[] OBX
		{
			get
			{
				return _obx;
			}
			set
			{
				_obx = value;
			}
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="366"> 
		///		<ExpectedInput>string array</ExpectedInput>
		///		<ExpectedOutput>string array</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="367"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// UniversalServiceID
		/// </summary>
		public string[] UniversalServiceID
		{
			get
			{
				return _universalServiceID;
			}
			set
			{
				_universalServiceID = value;
			}
		}


		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="368"> 
		///		<ExpectedInput>string array</ExpectedInput>
		///		<ExpectedOutput>string array</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="376"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// UniversalServiceID
		/// </summary>
		public string[] TimingQuantity
		{
			get
			{
				return _timingQuantity;
			}
			set
			{
				_timingQuantity = value;
			}
		}
	}
}
